// Career Consultations Website - Version 3 (HTML/CSS/JS)

const goals = [
  { 
    id: 'resume', 
    name: 'Резюме', 
    icon: 'icons/icon-file.svg',
    color: '#2563EB',
    description: 'Создание и оптимизация резюме',
    gradient: 'linear-gradient(135deg, #2563EB 0%, #3B82F6 100%)',
    duration: '60 минут',
    result: 'Готовое резюме',
    steps: ['Анализ опыта', 'Структурирование', 'Оптимизация']
  },
  { 
    id: 'interview', 
    name: 'Собеседование', 
    icon: 'icons/icon-users.svg',
    color: '#10B981',
    description: 'Подготовка к собеседованию',
    gradient: 'linear-gradient(135deg, #10B981 0%, #34D399 100%)',
    duration: '90 минут',
    result: 'Готовность к интервью',
    steps: ['Подготовка ответов', 'Практика', 'Стратегия']
  },
  { 
    id: 'career-change', 
    name: 'Смена профессии', 
    icon: 'icons/icon-target.svg',
    color: '#06B6D4',
    description: 'План перехода в новую сферу',
    gradient: 'linear-gradient(135deg, #06B6D4 0%, #22D3EE 100%)',
    duration: '120 минут',
    result: 'План действий',
    steps: ['Анализ навыков', 'План обучения', 'Стратегия перехода']
  },
  { 
    id: 'coaching', 
    name: 'Коучинг', 
    icon: 'icons/icon-trending.svg',
    color: '#8B5CF6',
    description: 'Долгосрочное карьерное развитие',
    gradient: 'linear-gradient(135deg, #8B5CF6 0%, #A78BFA 100%)',
    duration: '90 минут',
    result: 'Стратегия развития',
    steps: ['Целеполагание', 'Развитие навыков', 'Продвижение']
  }
]

const consultants = [
  { name: 'Елена Смирнова', experience: '12 лет', rating: 5, specialty: 'Резюме, Собеседование', clients: '2500+', success: '98%' },
  { name: 'Дмитрий Петров', experience: '15 лет', rating: 5, specialty: 'Собеседование, Коучинг', clients: '3200+', success: '96%' },
  { name: 'Анна Козлова', experience: '10 лет', rating: 5, specialty: 'Смена профессии, Коучинг', clients: '1800+', success: '94%' },
  { name: 'Михаил Волков', experience: '8 лет', rating: 5, specialty: 'Резюме, Карьерный рост', clients: '2100+', success: '97%' }
]

let currentSlide = 0
let slides = []
let dots = []
let slideInterval

function initPage() {
  const currentPage = window.location.pathname.includes('book.html') ? 'book' : 'index'
  
  if (currentPage === 'book') {
    initBookPage()
  } else {
    initIndexPage()
  }
  
  initSlider()
}

function initIndexPage() {
  loadSessions()
  updateProgress()
  
  setInterval(() => {
    loadSessions()
    updateProgress()
  }, 500)
}

function loadSessions() {
  const saved = localStorage.getItem('careerSessions')
  let sessions = []
  
  if (saved) {
    try {
      sessions = JSON.parse(saved)
    } catch (e) {
      sessions = []
    }
  } else {
    sessions = [
      {
        id: 1,
        consultant: 'Елена Смирнова',
        topic: 'Подготовка резюме',
        result: 'Резюме обновлено, получено 3 приглашения на собеседование',
        review: 'Отличная консультация, помогла структурировать опыт',
        rating: 5,
        date: '2025-03-25'
      },
      {
        id: 2,
        consultant: 'Дмитрий Петров',
        topic: 'Собеседование',
        result: 'Успешно прошел собеседование, получил оффер',
        review: 'Профессиональная подготовка, все советы пригодились',
        rating: 5,
        date: '2025-03-20'
      },
      {
        id: 3,
        consultant: 'Анна Козлова',
        topic: 'Смена профессии',
        result: 'Определен план перехода в новую сферу',
        review: 'Помогла понять, куда двигаться дальше',
        rating: 4,
        date: '2025-03-15'
      }
    ]
    localStorage.setItem('careerSessions', JSON.stringify(sessions))
  }
  
  const timelineContainer = document.getElementById('timelineContainer')
  if (timelineContainer) {
    if (sessions.length === 0) {
      timelineContainer.innerHTML = `
        <div class="empty-state-modern">
          <div class="empty-icon-modern">
            <img src="icons/hero-briefcase.svg" alt="Briefcase" width="80" height="80">
            <div class="empty-icon-ring"></div>
          </div>
          <h3 class="empty-title">Начните свой карьерный путь</h3>
          <p class="empty-text">Запишитесь на первую консультацию и увидите её здесь</p>
        </div>
      `
    } else {
      timelineContainer.innerHTML = sessions.map((session, index) => {
        const stars = Array.from({ length: 5 }, (_, i) => 
          `<span class="star" style="color: ${i < session.rating ? '#2563EB' : '#CBD5E1'}">★</span>`
        ).join('')
        
        const date = new Date(session.date).toLocaleDateString('ru-RU', { 
          day: 'numeric', 
          month: 'long', 
          year: 'numeric' 
        })
        
        const isLeft = index % 2 === 0
        
        return `
          <div class="timeline-item">
            <div class="timeline-marker">
              <div class="timeline-dot"></div>
              <div class="timeline-pulse"></div>
            </div>
            <div class="timeline-content ${isLeft ? 'left' : 'right'}">
              <div class="timeline-card">
                <div class="timeline-card-header">
                  <div class="timeline-topic-badge" style="background: linear-gradient(135deg, #2563EB 0%, #3B82F6 100%);">
                    ${session.topic}
                  </div>
                  <div class="timeline-rating">${stars}</div>
                </div>
                <div class="timeline-card-body">
                  <div class="timeline-consultant">
                    <img src="icons/icon-user.svg" alt="User" width="20" height="20">
                    <span>${session.consultant}</span>
                  </div>
                  <div class="timeline-result-box">
                    <img src="icons/icon-check.svg" alt="Check" width="20" height="20">
                    <div class="result-text-wrapper">
                      <span class="result-label">Достижение:</span>
                      <span class="result-text">${session.result}</span>
                    </div>
                  </div>
                  ${session.review ? `
                    <div class="timeline-review-box">
                      <span class="star" style="color: #F59E0B">★</span>
                      <span class="review-text">${session.review}</span>
                    </div>
                  ` : ''}
                </div>
                <div class="timeline-date">
                  <img src="icons/icon-calendar.svg" alt="Calendar" width="16" height="16">
                  ${date}
                </div>
              </div>
            </div>
          </div>
        `
      }).join('')
    }
  }
}

function updateProgress() {
  const saved = localStorage.getItem('careerSessions')
  let sessions = []
  
  if (saved) {
    try {
      sessions = JSON.parse(saved)
    } catch (e) {
      sessions = []
    }
  }
  
  const totalSessions = sessions.length
  const successfulResults = sessions.filter(s => 
    s.result.toLowerCase().includes('успешно') || 
    s.result.toLowerCase().includes('получен')
  ).length
  const averageRating = sessions.length > 0
    ? (sessions.reduce((sum, s) => sum + s.rating, 0) / sessions.length).toFixed(1)
    : 0
  const uniqueConsultants = new Set(sessions.map(s => s.consultant)).size
  
  const progressPercentage = totalSessions > 0 
    ? Math.min((successfulResults / totalSessions) * 100, 100) 
    : 0
  
  const progressPercentageEl = document.getElementById('progressPercentage')
  if (progressPercentageEl) {
    progressPercentageEl.textContent = Math.round(progressPercentage) + '%'
  }
  
  const progressBar = document.getElementById('progressBar')
  if (progressBar) {
    progressBar.style.width = progressPercentage + '%'
  }
  
  const progressStats = document.getElementById('progressStats')
  if (progressStats) {
    progressStats.innerHTML = `
      <div class="progress-stat-item">
        <div class="progress-stat-icon" style="background: rgba(37, 99, 235, 0.15);">
          <img src="icons/icon-briefcase.svg" alt="Briefcase" width="32" height="32">
        </div>
        <div class="progress-stat-content">
          <div class="progress-stat-value">${totalSessions}</div>
          <div class="progress-stat-label">Всего сессий</div>
        </div>
      </div>
      <div class="progress-stat-item">
        <div class="progress-stat-icon" style="background: rgba(16, 185, 129, 0.15);">
          <img src="icons/icon-trending.svg" alt="Trending" width="32" height="32">
        </div>
        <div class="progress-stat-content">
          <div class="progress-stat-value">${successfulResults}</div>
          <div class="progress-stat-label">Успешных результатов</div>
        </div>
      </div>
      <div class="progress-stat-item">
        <div class="progress-stat-icon" style="background: rgba(6, 182, 212, 0.15);">
          <img src="icons/icon-star.svg" alt="Star" width="32" height="32">
        </div>
        <div class="progress-stat-content">
          <div class="progress-stat-value">${averageRating}</div>
          <div class="progress-stat-label">Средняя оценка</div>
        </div>
      </div>
      <div class="progress-stat-item">
        <div class="progress-stat-icon" style="background: rgba(139, 92, 246, 0.15);">
          <img src="icons/icon-user.svg" alt="User" width="32" height="32">
        </div>
        <div class="progress-stat-content">
          <div class="progress-stat-value">${uniqueConsultants}</div>
          <div class="progress-stat-label">Консультантов</div>
        </div>
      </div>
    `
  }
}

function initBookPage() {
  const consultationForm = document.getElementById('consultationForm')
  const goalsShowcase = document.getElementById('goalsShowcase')
  const dateInput = document.getElementById('consultationDate')
  const successMessage = document.getElementById('successMessage')
  const consultantsShowcase = document.getElementById('consultantsShowcase')
  const submitButton = document.getElementById('submitButton')

  let selectedGoal = null

  if (goalsShowcase) {
    goalsShowcase.innerHTML = goals.map(goal => `
      <button type="button" class="goal-showcase-card" data-id="${goal.id}" style="border-color: rgba(37, 99, 235, 0.2); background: white;">
        <div class="goal-showcase-header">
          <div class="goal-showcase-icon-wrapper" style="background: ${goal.color}15;">
            <img src="${goal.icon}" alt="${goal.name}" width="44" height="44">
          </div>
        </div>
        <div class="goal-showcase-content">
          <h3 class="goal-showcase-name" style="color: var(--text);">${goal.name}</h3>
          <p class="goal-showcase-description" style="color: var(--text-light);">${goal.description}</p>
          <div class="goal-showcase-steps">
            ${goal.steps.map(step => `
              <div class="goal-step-item" style="color: var(--text-light);">
                <span>⚡</span>
                <span>${step}</span>
              </div>
            `).join('')}
          </div>
          <div class="goal-showcase-meta">
            <div class="goal-meta-item" style="color: var(--text-light);">
              <img src="icons/icon-clock.svg" alt="Clock" width="18" height="18">
              <span>${goal.duration}</span>
            </div>
            <div class="goal-meta-item" style="color: var(--text-light);">
              <img src="icons/icon-check.svg" alt="Check" width="18" height="18">
              <span>${goal.result}</span>
            </div>
          </div>
        </div>
      </button>
    `).join('')

    goalsShowcase.querySelectorAll('.goal-showcase-card').forEach(card => {
      card.addEventListener('click', () => {
        goalsShowcase.querySelectorAll('.goal-showcase-card').forEach(c => {
          c.classList.remove('selected')
          c.style.background = 'white'
          c.style.color = 'var(--text)'
        })
        card.classList.add('selected')
        selectedGoal = card.dataset.id
        const goalData = goals.find(g => g.id === selectedGoal)
        if (goalData) {
          card.style.background = goalData.gradient
          card.style.color = 'white'
          card.querySelector('.goal-showcase-name').style.color = 'white'
          card.querySelector('.goal-showcase-description').style.color = 'rgba(255, 255, 255, 0.9)'
          card.querySelectorAll('.goal-step-item').forEach(item => {
            item.style.color = 'rgba(255, 255, 255, 0.9)'
          })
          card.querySelectorAll('.goal-meta-item').forEach(item => {
            item.style.color = 'rgba(255, 255, 255, 0.9)'
          })
          card.querySelector('.goal-showcase-icon-wrapper').style.background = 'rgba(255, 255, 255, 0.25)'
        }
        
        if (submitButton) {
          submitButton.disabled = !selectedGoal || !dateInput.value
        }
      })
    })
  }

  if (dateInput) {
    const today = new Date()
    const maxDate = new Date()
    maxDate.setDate(maxDate.getDate() + 90)
    
    dateInput.min = today.toISOString().split('T')[0]
    dateInput.max = maxDate.toISOString().split('T')[0]
    
    dateInput.addEventListener('change', () => {
      if (submitButton) {
        submitButton.disabled = !selectedGoal || !dateInput.value
      }
    })
  }

  if (consultationForm) {
    consultationForm.addEventListener('submit', (e) => {
      e.preventDefault()
      
      if (selectedGoal && dateInput && dateInput.value) {
        const savedSessions = localStorage.getItem('careerSessions')
        const sessions = savedSessions ? JSON.parse(savedSessions) : []
        
        const goalData = goals.find(g => g.id === selectedGoal)
        const randomConsultant = consultants[Math.floor(Math.random() * consultants.length)]
        
        const newSession = {
          id: Date.now(),
          consultant: randomConsultant.name,
          topic: goalData.name,
          result: `Консультация запланирована. Ожидаемый результат: ${goalData.result}`,
          review: '',
          rating: 0,
          date: dateInput.value
        }

        sessions.push(newSession)
        localStorage.setItem('careerSessions', JSON.stringify(sessions))
        
        if (successMessage) {
          successMessage.style.display = 'flex'
          setTimeout(() => {
            window.location.href = 'index.html'
          }, 2000)
        }
      }
    })
  }

  if (consultantsShowcase) {
    consultantsShowcase.innerHTML = consultants.map(consultant => {
      const stars = Array.from({ length: 5 }, () => '★').join('')
      return `
        <div class="consultant-showcase-card">
          <div class="consultant-showcase-avatar">
            <img src="icons/icon-user.svg" alt="${consultant.name}" width="48" height="48">
            <div class="consultant-avatar-ring"></div>
            <div class="consultant-success-badge">
              <span>${consultant.success}</span>
            </div>
          </div>
          <div class="consultant-showcase-content">
            <h3 class="consultant-name">${consultant.name}</h3>
            <p class="consultant-experience">${consultant.experience} опыта</p>
            <div class="consultant-rating">
              ${Array.from({ length: 5 }, () => '<span style="color: #2563EB;">★</span>').join('')}
            </div>
            <p class="consultant-specialty">${consultant.specialty}</p>
            <div class="consultant-stats">
              <div class="consultant-stat-item">
                <img src="icons/icon-users.svg" alt="Users" width="18" height="18">
                <span>${consultant.clients} клиентов</span>
              </div>
              <div class="consultant-stat-item">
                <img src="icons/icon-trending.svg" alt="Trending" width="18" height="18">
                <span>${consultant.success} успеха</span>
              </div>
            </div>
          </div>
        </div>
      `
    }).join('')
  }
}

function initSlider() {
  const slider = document.getElementById('slider')
  if (!slider) return

  slides = Array.from(slider.querySelectorAll('.slide'))
  dots = Array.from(slider.querySelectorAll('.dot'))
  const prevBtn = document.getElementById('sliderPrev')
  const nextBtn = document.getElementById('sliderNext')

  if (slides.length === 0) return

  function showSlide(n) {
    slides.forEach(slide => slide.classList.remove('active'))
    dots.forEach(dot => dot.classList.remove('active'))
    
    currentSlide = (n + slides.length) % slides.length
    
    slides[currentSlide].classList.add('active')
    if (dots[currentSlide]) dots[currentSlide].classList.add('active')
  }

  function nextSlide() {
    showSlide(currentSlide + 1)
  }

  function prevSlide() {
    showSlide(currentSlide - 1)
  }

  if (nextBtn) {
    nextBtn.addEventListener('click', () => {
      clearInterval(slideInterval)
      nextSlide()
      startSlider()
    })
  }

  if (prevBtn) {
    prevBtn.addEventListener('click', () => {
      clearInterval(slideInterval)
      prevSlide()
      startSlider()
    })
  }

  dots.forEach((dot, index) => {
    dot.addEventListener('click', () => {
      clearInterval(slideInterval)
      showSlide(index)
      startSlider()
    })
  })

  function startSlider() {
    slideInterval = setInterval(nextSlide, 3000)
  }

  startSlider()
}

document.addEventListener('DOMContentLoaded', initPage)

